#!/usr/bin/env bash
# install-apps.sh
# Automatisierte Installation unter Linux (apt/dnf/pacman-Erkennung)
# Ausführung: sudo bash install-apps.sh

set -o errexit
set -o pipefail
set -o nounset

LOG_FILE="./install-log_$(date +%Y%m%d_%H%M%S).log"
exec > >(tee -a "$LOG_FILE") 2>&1

echo ">>> Automatische Installation startet..."

PKG=""
if command -v apt >/dev/null 2>&1; then
  PKG="apt"
elif command -v dnf >/dev/null 2>&1; then
  PKG="dnf"
elif command -v pacman >/dev/null 2>&1; then
  PKG="pacman"
else
  echo "[FEHLER] Kein unterstützter Paketmanager gefunden (apt/dnf/pacman)."
  exit 1
fi

echo "--- Paketmanager: $PKG"

case "$PKG" in
  apt) apt update -y ;;
  dnf) dnf makecache -y ;;
  pacman) pacman -Sy --noconfirm ;;
esac

install_pkg() {
  local name="$1"
  shift
  local pkgs=("$@")

  echo "--- $name wird installiert..."
  if [[ "$PKG" == "apt" ]]; then
    if ! apt install -y "${pkgs[@]}"; then
      echo "[FEHLER] $name konnte nicht installiert werden."
      return 1
    fi
  elif [[ "$PKG" == "dnf" ]]; then
    if ! dnf install -y "${pkgs[@]}"; then
      echo "[FEHLER] $name konnte nicht installiert werden."
      return 1
    fi
  elif [[ "$PKG" == "pacman" ]]; then
    if ! pacman -S --noconfirm "${pkgs[@]}"; then
      echo "[FEHLER] $name konnte nicht installiert werden."
      return 1
    fi
  fi
  echo "[OK] $name wurde installiert."
}

# Beispielpakete
install_pkg "Firefox" firefox || true

if [[ "$PKG" == "apt" ]]; then
  install_pkg "Chromium" chromium-browser || install_pkg "Chromium" chromium || true
elif [[ "$PKG" == "dnf" ]]; then
  install_pkg "Chromium" chromium || true
elif [[ "$PKG" == "pacman" ]]; then
  install_pkg "Chromium" chromium || true
fi

if [[ "$PKG" == "apt" ]]; then
  install_pkg "7-Zip (p7zip-full)" p7zip-full || true
else
  install_pkg "p7zip" p7zip || true
fi

install_pkg "VLC" vlc || true

# VS Code bzw. Code-OSS
if [[ "$PKG" == "apt" ]]; then
  if apt-cache show code >/dev/null 2>&1; then
    install_pkg "Visual Studio Code" code || true
  else
    install_pkg "VS Code (OSS)" code-oss || true
  fi
elif [[ "$PKG" == "dnf" ]]; then
  if dnf info code >/dev/null 2>&1; then
    install_pkg "Visual Studio Code" code || true
  else
    install_pkg "VS Code (OSS)" code-oss || true
  fi
elif [[ "$PKG" == "pacman" ]]; then
  if pacman -Si code >/dev/null 2>&1; then
    install_pkg "Visual Studio Code" code || true
  else
    install_pkg "VS Code (OSS)" code-oss || true
  fi
fi

echo "=== Fertig. Log: $LOG_FILE"
