﻿# Install-Apps.ps1
# Einfache automatische Software-Installation mit winget
# Ausführung: PowerShell als Administrator öffnen → .\Install-Apps.ps1

# ============ Grundeinstellungen & Logging ============

$LogPath = "$PSScriptRoot\install-log_{0}.txt" -f (Get-Date -Format 'yyyyMMdd_HHmmss')
Start-Transcript -Path $LogPath -Append | Out-Null

function Info($msg) { Write-Host $msg }
function Ok($msg)   { Write-Host $msg -ForegroundColor Green }
function Warn($msg) { Write-Host $msg -ForegroundColor Yellow }
function Err($msg)  { Write-Host $msg -ForegroundColor Red }

Info ">>> Automatische Installation startet..."
Info ("Log-Datei: {0}" -f $LogPath)
Info ""

# ============ Prüfen, ob winget vorhanden ist ============

if (-not (Get-Command winget -ErrorAction SilentlyContinue)) {
    Err "ERROR: 'winget' wurde nicht gefunden. Bitte App Installer / winget installieren."
    Stop-Transcript | Out-Null
    exit 1
}

# Optional: kurzer Testlauf, ob winget grundsätzlich reagiert
try {
    Info "winget wird kurz getestet..."
    winget --version | Out-Null
    if ($LASTEXITCODE -ne 0) {
        Warn "Warnung: winget liefert einen Fehlercode beim Test."
    }
} catch {
    Err "ERROR: Beim Test von winget ist ein Fehler aufgetreten: $($_.Exception.Message)"
}

# ============ Hilfsfunktionen ============

function Test-Installiert {
    param(
        [Parameter(Mandatory=$true)][string]$Id
    )
    try {
        # Interaktivität deaktivieren, nur Abfrage
        $output = winget list --id $Id --exact --disable-interactivity 2>$null
        if ($LASTEXITCODE -eq 0 -and $output -match $Id) {
            return $true
        } else {
            return $false
        }
    } catch {
        return $false
    }
}

function Install-App {
    param(
        [Parameter(Mandatory=$true)][string]$Name,
        [Parameter(Mandatory=$true)][string]$Id
    )

    Info ("--- {0} wird geprüft..." -f $Name)

    $installiert = Test-Installiert -Id $Id

    if ($installiert) {
        Ok ("[OK] {0} ist bereits installiert." -f $Name)
        return
    }

    Info ("{0} ist nicht installiert → Installation wird gestartet..." -f $Name)

    try {
        # Silent-Install mit deaktivierter Interaktivität
        winget install --id $Id --exact `
            --silent `
            --accept-source-agreements `
            --accept-package-agreements `
            --disable-interactivity

        if ($LASTEXITCODE -eq 0) {
            Ok ("[OK] {0} wurde erfolgreich installiert." -f $Name)
        } else {
            Warn ("[FEHLER] {0} konnte nicht installiert werden (ExitCode {1})." -f $Name, $LASTEXITCODE)
        }
    } catch {
        Err ("[FEHLER] Ausnahme beim Installieren von {0}: {1}" -f $Name, $_.Exception.Message)
    }
}

# ============ Hier die zu installierenden Apps definieren ============

# Du kannst diese Liste später erweitern:
# @{ Name = "App-Name"; Id = "Hersteller.AppId" }

$apps = @(
    @{ Name = "Mozilla Firefox";        Id = "Mozilla.Firefox" }
    @{ Name = "Mozilla Firefox ESR";    Id = "Mozilla.Firefox.ESR" }

    # Browser alternatif
    @{ Name = "Google Chrome";          Id = "Google.Chrome" }

    # Password Manager
    @{ Name = "KeePassXC";              Id = "KeePassXCTeam.KeePassXC" }

    # ZIP / Archiv
    @{ Name = "7-Zip";                  Id = "7zip.7zip" }

    # Editor
    @{ Name = "Notepad++";              Id = "Notepad++.Notepad++" }

    # PDF Viewer
    @{ Name = "Adobe Acrobat Reader";   Id = "Adobe.Acrobat.Reader.64-bit" }

    # Video/Audio
    @{ Name = "VLC Media Player";       Id = "VideoLAN.VLC" }
)

# ============ Installation ausführen ============

foreach ($app in $apps) {
    Install-App -Name $app.Name -Id $app.Id
}

Info ""
Info ">>> Fertig. Details siehe Log-Datei:"
Info $LogPath

Stop-Transcript | Out-Null
